% zDisplayNT(File,NTList,ViewParam) is a general-purpose nucleotide
% plotting program.  It can be called in several ways, for example,
% zDisplayNT('rr0033_23S',{'27','28','29'},ViewParam), and it will load the
% named datafile and plot the nucleotides by nucleotide number.  Or, if
% data files have already been loaded, one can use zDisplayNT(File(1),[32
% 34 35],ViewParam) to plot the nucleotides in File(1) having indices 32,
% 34, and 35.  Defaults for ViewParam are defined in zDisplayNT; see there
% for the fields of ViewParam.
% One can also use ranges of nucleotide numbers, as in
% zDisplayNT('rr0033_23S',{'2548:2555','2557','2559:2566'},VP);

function [ViewPar] = zDisplayNT(File,NTList,ViewParam)

% set default values

VP.Sugar     = 1;

CurrentView=get(gca,'view');
VP.az=CurrentView(1);
VP.el=CurrentView(2);

VP.LineStyle = '-';              % default - thick solid lines
VP.LineThickness = '2';          
VP.AtOrigin  = 0;                % rotate all so first is at origin
VP.Title     = 1;                % title with nucleotide numbers, filename
VP.Grid      = 0;                % add a grid to the graph
VP.FontSize  = 10;               % will use Matlab's default unless overridden
VP.Rotation  = eye(3);
VP.Shift     = zeros(1,3);
VP.LabelBases = 1;

if nargin == 2,
  ViewParam = VP;
end

% replace defaults with defined values

if isfield(ViewParam,'Sugar'),
  VP.Sugar = ViewParam.Sugar;
end

if isfield(ViewParam,'az'),
  VP.az = ViewParam.az;
end

if isfield(ViewParam,'el'),
  VP.el = ViewParam.el;
end

if isfield(ViewParam,'LineStyle'),
  VP.LineStyle = ViewParam.LineStyle;
end

if isfield(ViewParam,'LineThickness'),
  VP.LineThickness = ViewParam.LineThickness;
end

if isfield(ViewParam,'AtOrigin'),
  VP.AtOrigin = ViewParam.AtOrigin;
end

if isfield(ViewParam,'Title'),
  VP.Title = ViewParam.Title;
end

if isfield(ViewParam,'Grid'),
  VP.Grid = ViewParam.Grid;
end

if isfield(ViewParam,'FontSize'),
  h=gca;
  set(h,'FontSize',ViewParam.FontSize);
end

if isfield(ViewParam,'Rotation'),
  VP.Rotation = ViewParam.Rotation;
end

if isfield(ViewParam,'Shift'),
  VP.Shift = ViewParam.Shift;
end

if isfield(ViewParam,'LabelBases'),
  VP.Shift = ViewParam.LabelBases;
end

% if File is a text string (filename), load the file and display
if strcmp(class(File),'char'),
  Filename = File;
  File = zGetNTData(Filename,0);
end

% if NTList is a cell array of numbers, look up the indices

if strcmp(class(NTList),'char'),
  NTList = {NTList};
end

if strcmp(class(NTList),'cell'),
  Indices = zIndexLookup(File,NTList);
else
  Indices = NTList;
end

% plot the nucleotides, at the origin or in original positions

set(gcf,'Renderer','OpenGL');

if VP.AtOrigin == 1,
  R = File.NT(Indices(1)).Rot;             % Rotation matrix for first base
  S = File.NT(Indices(1)).Fit(1,:);        % Location of glycosidic atom
else
  R = VP.Rotation;
  S = VP.Shift;
end

for j=1:length(Indices),                 % Loop through all nucleotides
  k = Indices(j);                        % index of current nucleotide
  zPlotOneNTRotated(File.NT(k),VP,R,S);
end

% connect the sugars, if sugars are being plotted

if VP.Sugar > 0,
  [D,i] = sort(Indices);
  for j=1:(length(Indices)-1),
    if D(j+1)-D(j) == 1,
      A = [File.NT(D(j)).Sugar(5,:); File.NT(D(j+1)).Sugar(10,:)];
      AA = (A - ones(2,3)*S) * R;                                           %Ali ones(2,3) instead of ones(2,1)
      plot3(AA(:,1), AA(:,2), AA(:,3),'w','LineWidth',2,'LineStyle','-');   %'w' instead of 'k'
    end
  end
end

ind=strfind(File.Filename,filesep);
Title1 = File.Filename(ind(end)+1:end);
Title1 = strrep(Title1,'_','\_');
% % % if length(Indices)<10
% % %     Title2 = strcat(File.NT(Indices(1)).Base,File.NT(Indices(1)).Number);
% % %     for j=2:length(Indices),
% % %         nt = File.NT(Indices(j));
% % %         Title2 = strcat(Title2,'-',nt.Base,nt.Number);
% % %     end
% % % else
% % %     Title2=[];
% % % end
% % % Title=strvcat(Title1,Title2)
title(Title1,'Color','w');

axis equal
view(VP.az,VP.el);


if VP.Grid == 1,
  grid on
else
  grid off
end

