% zReadBermanFile reads in previously created expert classifications.
% These are stored in the following fields of File:

% BermanClass    m x 1 array of expert classification categories
% Comment      m row cell array of comments on each pair
% CI           NumNT x NumNT matrix of indices to the previous two arrays

% Thus, if you want to know what the expert classification is of the
% interaction between bases with indices p and q, you look at
% Comment{CI(p,q)} and BermanClass(CI(p,q))

function [File] = zReadBermanFile(File)

BermanClass = [];                         % hand category read from file
BI          = sparse(File.NumNT,File.NumNT);        % matrix of indices

if exist(strcat(File.Filename,'_Berman.xls'),'file') > 0,
  a = textread('rr0033_Berman.csv', '%s','delimiter','\n');

  for i=1:length(a(:,1)),
    s = a{i};
    u = strfind(s,'_');
    if length(u) > 1,
      c = strfind(s,':');
      h = strfind(s,',');
      p = strfind(s,'+');    % just in case there are extra letters
      Chain1{i}   = s((h(1)+1):(u(1)-1));
      NuclNum1{i} = s((u(1)+2):(c-1));
      Chain2{i}   = s((u(2)+1):(h(2)-1));
      NuclNum2{i} = s((c+2):u(2)-1);
      class = str2num(s((h(2)+1):(h(3)-1)));
      if class > 0,
        BermanClass(i) = class;
      else
        BermanClass(i) = 15;        % something, but unknown
      end
    else
      fprintf('Trouble reading %s\n',s);
    end
  end

  Numbers   = cat(1,{File.NT(:).Number});

  for a=1:length(BermanClass),
    p = find(ismember(Numbers,NuclNum1{a}));  % get index of first nucleotide
    q = find(ismember(Numbers,NuclNum2{a}));  % get index of second nucleotide

    if length(p) > 1,
      for j = 1:length(p),
        if strcmp(Chain1{a},File.NT(p(j)).Chain) == 1,
          p(1) = p(j);
        end
      end
      p = p(1);
    end
 
    if length(q) > 1,
      for j = 1:length(q),
        if strcmp(Chain1{a},File.NT(q(j)).Chain) == 1,
          q(1) = q(j);
        end
      end
      q = q(1);
    end
 
    if (p>0) & (q>0),
      if BI(p,q) > 0,
        fprintf('Nucleotides %5s and %5s ',NuclNum1{a}, NuclNum2{a});
        fprintf('are Berman classified twice.\n');
      end
      BI(p,q) = a;                        % points pair to row of Comment
      BI(q,p) = a;                        % points pair to row of Comment
      if (File.Distance(p,q) == 0),
        fprintf('Nucleotides %5s and %5s ',NuclNum1{a}, NuclNum2{a});
        fprintf('are more than 15 Angstroms apart.\n');
      end
    else
      fprintf('Could not find one of nucleotides ');
      fprintf('%s and %s in Berman file.\n', NuclNum1{a}, NuclNum2{a});
    end
  end
  fprintf('Read %s\n', [File.Filename 'Berman.xls']);
end

% Add fields to File ---------------------------------------------------------

File.BermanClass = BermanClass;
File.BI        = BI;
